import { ActionFormData, MessageFormData, ModalFormData } from "@minecraft/server-ui";
import {SchoolClass, SchoolSubject, createSchoolClass} from "./Classes";
import { Student, Human, addPersonalToClass } from "./Humans";
import { Player, world } from "@minecraft/server";

/**
 * Represents a school
 */
export class School{
    /**
     * 
     * @param {string} name This school's name
     * @param {Array<SchoolSubject>} perfil This school's perfil
     */
    constructor(name, perfil){
        /**
         * @type {string}
         * @description This school's name
         */
        this.name = name;
        /**
         * @type {Array<Subject>}
         * @description This school's perfil
         */
        this.perfil = perfil;
        /**
         * @type {Array<Human>}
         * @description Personal working at school
         */
        this.personal = [];
        /**
         * @type {Array<Teacher>}
         * @description Teachers working at school
         */
        this.teachers = [];
        /**
         * @type {Array<Student>}
         * @description Students learning at school
         */
        this.students = [];
        /**
         * @type {Human}
         * @description The headmaster of the school
         */
        this.headmaster = [];
        /**
         * @type {Array<SchoolClass>}
         * @description The classes at school
         */
        this.classes = [];
        this.oldStudents = [];
        this.oldPersonal = [];
    };
    /**
     * @type {Array<School>}
     * @description List with all schools registered
     */
    static schools = []
    /**
     * Registers this school, making it able to be found using School.lookForSchool()
     * @returns {void}
     */
    registerSchool(){
        for(let s = 0; s < School.schools.length; s++){ if(School.schools[s].name == this.name){ console.warn("School with name:"+this.name+", is already registered!"); return; }; };
        School.schools.push(this); console.warn("Neew school with name: "+this.name);
    };
    /**
     * Looks for a registerd school
     * @param {string} name The school's name to look for
     * @returns {School|undefined}
     */
    static lookForSchool(name){
        for(let s = 0; s < School.schools.length; s++){ if(School.schools[s].name == name){ return School.schools[s]; }; };
        console.warn("School with name: "+name+", not found! Either not registered or wrong name value!");
        return undefined;
    };
    /**
     * Registers a SchoolClass at this school
     * @param {SchoolClass} schoolClass 
     * @returns {number|undefined} The schoolClassIdentifier of the class at this school
     */
    registerClass(schoolClass){
        for(let cl = 0; cl < this.classes.length; cl++){ if(this.classes[cl].identifier == schoolClass.identifier){ console.warn("SchoolClass with identifier: "+schoolClass.identifier+", already in school!"); return undefined; }; };
        schoolClass.schoolClassIdentifier = this.classes.length; this.classes.push(schoolClass); 
        return schoolClass.schoolClassIdentifier;
    };
    /**
     * Adds Students into this school and into a class
     * @param {Array<Student>} students List of Students that want to join this school
     * @param {string|number} classes Class for the students to join; 'Random_' will make the students join random School Classes
     */
    addStudents(students, classes){
        /**
         * @type {Array<Student>}
         */
        let denied = [];
        for(let i = 0; i<students.length; i++){
            for(let s = 0; s<this.students.length; s++){ if(this.students[s].identifier==students[i].identifier){ console.warn(`Student with Id ${students[i].identifier} denied`); denied.push(students[i]); }; };
            if(!denied.includes(students[i])){ if(classes == "Random_"){ let clas = Math.round(Math.random()*this.classes.length%this.classes.length); this.classes[clas].newStudents([students[i]]); console.warn("Student joined class Id"+clas); }else { this.classes[classes].newStudents([students[i]]); console.warn(`Student joined given class: ${classes}`); } };
        };
    };
    /**
     * Searches this school for a Student with a given Id
     * @param {string|number} id The Id of a Student to look for
     * @returns {Object|void}
     */
    lookForStudent(id){
        if(!Human.lookForHuman(id)){console.warn("Human does not exist!"); return;}
        if(!this.students.includes(Human.lookForHuman(id))){console.warn("Student not in this School!"); return;};
        for(let clas = 0; clas < this.classes.length; clas++){ if(this.classes[clas].students.includes(Human.lookForHuman(id))){ for(let s = 0; s < this.classes[clas].students.length; s++){ if(this.classes[clas].students[s].identifier==id){ return {foundStudent: this.classes[clas].students[s], inClass: this.classes[clas]}; } } } };
        console.warn("Student not found in any class!");
        return; 
    };
    /**
     * Removes a Student from the entire school
     * @param {string|number} id The Id of the student to expulse 
     */
    expulseStudent(id){
        if(!Human.lookForHuman(id)){console.warn("Human does not exist!"); return;}
        if(!this.students.includes(Human.lookForHuman(id))){console.warn("Student not in this School!"); return;};
        for(let clas = 0; clas < this.classes.length; clas++){ if(this.classes[clas].students.includes(Human.lookForHuman(id))){ for(let s = 0; s < this.classes[clas].students.length; s++){ if(this.classes[clas].students[s].identifier==id){this.classes[clas].students.splice(s, 1) ;return;} } } };
        console.warn("Student not found in any class!");
        return;
    };
    /**
     * Updates the School's students and personal Objects
     */
    static updateOld(){
        for(let s = 0; School.schools.length; s++){
            world.sendMessage(JSON.stringify({...School.schools[s]}));
            for(let personal = 0; personal<School.schools[s].oldPersonal.length; personal++){
                if(Human.lookForHuman(School.schools[s].oldPersonal[personal]) == undefined){
                    continue;
                }else {
                    School.schools[s].personal.push(Human.lookForHuman(School.schools[s].oldPersonal[personal]));
                    School.schools[s].oldPersonal.splice(personal, 1);
                };
            };
            for(let student = 0; student < School.schools[s].oldStudents.length; student++){
                if(Human.lookForHuman(School.schools[s].oldStudents[student])==undefined){
                    continue;
                }else{
                    School.schools[s].students.push(Human.lookForHuman(School.schools[s].oldStudents[student]));
                    School.schools[s].oldStudents.splice(student, 1);
                };
            }
        }
    }
};
/**
 * Shows a Menu with All School Information
 * @param {Player} player The player to show the menus
 */
export function showSchool(player){
    let main = new ActionFormData(); main.title("School"); main.body("Select a School");
    if(School.schools.length==0){
        main.button("NO Schools Availible!", "textures/blocks/barrier");
    }else {
        for(let s = 0; s < School.schools.length; s++){
            console.warn(s+" _"+School.schools[s].name)
            main.button(School.schools[s].name);
        }
    };
    main.show(player).then((res1)=>{
        if(res1.canceled ||School.schools.length==0){return;};
        let schoolData = new ActionFormData(); schoolData.body("Select what to check"); schoolData.title("School");
        schoolData.button("Personal"); schoolData.button("Classes"); schoolData.button("Extra Information");
        schoolData.show(player).then((res2)=>{
            if(res2.canceled){return;};
            /**
             * @type {School}
             */
            let schol = School.schools[res1.selection];
            switch(res2.selection){
                case 0: let responseM = new ActionFormData(); responseM.title("School"); responseM.body("List of Personal");let i = 0; for(let p = 0; p<schol.personal.length; p++){responseM.button(schol.personal[p].name+" "+schol.personal[p].familyName); i++}; if(schol.personal.length==0){responseM.button("No Personal in School!");i=1};responseM.button("Add Personal"); responseM.show(player).then((resp)=>{if(resp.canceled){return;}; if(resp.selection==i){addPersonalToClass(player, schol)}else {return;};}); break;
                case 1: let responseN = new ActionFormData(); responseN.title("School"); responseN.body("List of Classes");let j = 0; for(let cl = 0; cl< schol.classes.length; cl++){responseN.button(schol.classes[cl].grade+""+schol.classes[cl].letter); j++}; if(schol.classes.length==0){responseN.button("No Classes in School!");j=1};responseN.button("Create Class"); responseN.show(player).then((resp)=>{if(resp.selection==j){createSchoolClass(player, schol);}; player.sendMessage(schol.classes[resp.selection].students.toString()+"\nIf you wish to add Students, run jo:addStudent.");if(resp.canceled){return;}; if(resp.selection==j){player.runCommand("scriptevent jo:classCreate run")}else {return;};}); break;
                case 2: let responseO = new MessageFormData(); responseO.title("School"); responseO.body(`Extra Information about this school\n name: ${schol.name}\n  perfil: ${schol.perfil.toString()}`); responseO.button1("Okay"); responseO.button2("Close"); responseO.show(player).then((resp)=>{return;}); break;
                default: break;
            };
            return;
        });
    });
};
/**
 * Allows a Player to create a new School in-game
 * @param {Player} player The player to build the school
 * @returns {undefined|Object}
 */
export function createSchool(player){
    let creator = new ModalFormData(); creator.title("School"); creator.textField("New School's name", "Name"); creator.dropdown("New School's main perfil", SchoolSubject.AllTogether.split(", ")); creator.dropdown("New School's secondary perfil", SchoolSubject.AllTogether.split(", "));
    creator.show(player).then((response)=>{
        if(response.canceled){return undefined;};
        if(response.formValues[0]==""){return undefined; };
        let name = response.formValues[0]; let perfil = [SchoolSubject.AllTogether.split(", ")[response.formValues[1]], SchoolSubject.AllTogether.split(", ")[response.formValues[2]]];
        let schoolo = new School(name, perfil);
        schoolo.registerSchool();
        player.sendMessage("Created new School! "+name);
    });
};
